<?php
/**
 * Plugin Name: Seamless Videos Player
 * Description: 複数のMP4ファイルを連結して１つの動画にしてシームレスに再生するプラグイン。[svp_player videos="url,url,url"] の形でショートコードを挿入すると可能。
 * Version: 1.2
 * Author: <a href="https://yukimal.com">https://yukimal.com</a>
 */

if (!defined('ABSPATH')) {
    exit; // 直接アクセスを防ぐ
}

// ショートコードを追加
function svp_video_player_shortcode($atts) {
    $atts = shortcode_atts([
        'videos' => '', // コンマ区切りの動画URL
    ], $atts);

    $video_urls = array_map('trim', explode(',', $atts['videos']));
    if (empty($video_urls[0])) return '動画が指定されていません。';

    ob_start();
    ?>
    <div id="svp-player-container">
        <div id="svp-video-wrapper" style="position: relative; width: 100%; max-width: 800px; margin: auto;">
            <video id="svp-player" playsinline>
                <source id="svp-source" src="<?php echo esc_url($video_urls[0]); ?>" type="video/mp4">
            </video>
            <div id="svp-controls">
                <button id="svp-play-pause">▶</button>
                <input type="range" id="svp-seekbar" min="0" step="0.1" value="0">
                <button id="svp-fullscreen">⛶</button>
            </div>
        </div>
        <div id="svp-chapters">
            <?php foreach ($video_urls as $index => $video) : ?>
                <button class="svp-chapter" data-index="<?php echo $index; ?>">
                    <?php echo $index + 1; ?>
                </button>
            <?php endforeach; ?>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const video = document.getElementById('svp-player');
            const source = document.getElementById('svp-source');
            const playPauseButton = document.getElementById('svp-play-pause');
            const fullscreenButton = document.getElementById('svp-fullscreen');
            const videoWrapper = document.getElementById('svp-video-wrapper');
            const controls = document.getElementById('svp-controls');
            const chapters = document.querySelectorAll('.svp-chapter');
            const seekbar = document.getElementById('svp-seekbar');
            let videoIndex = 0;
            const videoList = <?php echo json_encode($video_urls); ?>;
            const videoDurations = [];
            let totalDuration = 0;
            
            function loadVideoDurations(index = 0) {
                if (index >= videoList.length) {
                    totalDuration = videoDurations.reduce((a, b) => a + b, 0);
                    seekbar.max = totalDuration;
                    return;
                }
                let tempVideo = document.createElement('video');
                tempVideo.src = videoList[index];
                tempVideo.addEventListener('loadedmetadata', function() {
                    videoDurations[index] = tempVideo.duration;
                    loadVideoDurations(index + 1);
                });
            }
            loadVideoDurations();
            
            function getCurrentTimeInPlaylist() {
                let time = 0;
                for (let i = 0; i < videoIndex; i++) {
                    time += videoDurations[i];
                }
                return time + video.currentTime;
            }

            function seekToTime(time) {
                let accumulatedTime = 0;
                for (let i = 0; i < videoList.length; i++) {
                    if (accumulatedTime + videoDurations[i] >= time) {
                        videoIndex = i;
                        source.src = videoList[i];
                        video.load();
                        video.currentTime = time - accumulatedTime;
                        video.play();
                        updateChapterStyles();
                        return;
                    }
                    accumulatedTime += videoDurations[i];
                }
            }
            
            function changeVideo(index) {
                video.pause();
                source.src = videoList[index];
                video.load();
                video.play();
                videoIndex = index;
                updateChapterStyles();
            }

            function updateChapterStyles() {
                chapters.forEach((btn, idx) => {
                    btn.style.backgroundColor = (idx === videoIndex) ? 'orange' : '';
                });
            }
            
            chapters.forEach((btn, index) => {
                btn.addEventListener('click', () => changeVideo(index));
            });

            video.addEventListener('timeupdate', function() {
                seekbar.value = getCurrentTimeInPlaylist();
            });
            
            seekbar.addEventListener('input', function() {
                seekToTime(parseFloat(seekbar.value));
            });

            video.addEventListener('ended', function() {
                if (videoIndex < videoList.length - 1) {
                    changeVideo(videoIndex + 1);
                }
            });

            playPauseButton.addEventListener('click', function() {
                if (video.paused) {
                    video.play();
                    playPauseButton.textContent = '❚❚';
                } else {
                    video.pause();
                    playPauseButton.textContent = '▶';
                }
            });

            fullscreenButton.addEventListener('click', function() {
                if (!document.fullscreenElement) {
                    if (videoWrapper.requestFullscreen) {
                        videoWrapper.requestFullscreen();
                    } else if (videoWrapper.mozRequestFullScreen) {
                        videoWrapper.mozRequestFullScreen();
                    } else if (videoWrapper.webkitRequestFullscreen) {
                        videoWrapper.webkitRequestFullscreen();
                    } else if (videoWrapper.msRequestFullscreen) {
                        videoWrapper.msRequestFullscreen();
                    }
                } else {
                    if (document.exitFullscreen) {
                        document.exitFullscreen();
                    }
                }
            });

            document.addEventListener('fullscreenchange', function() {
                if (document.fullscreenElement) {
                    controls.style.position = 'fixed';
                    controls.style.bottom = '5px';
                    controls.style.width = '100%';
                } else {
                    controls.style.position = 'absolute';
                }
            });

            updateChapterStyles();
        });
    </script>
    <style>
        #svp-player-container {
            text-align: center;
        }
        video {
            width: 100%;
            max-height: 90vh;
            display: block;
            background: black;
        }
        #svp-controls {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 7px;
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            padding: 5px;
        }
        #svp-fullscreen{
            background: rgba(20,20,20,0.3);
            color: #ddd;
        }
        #svp-play-pause{
            background: rgba(20,20,20,0.3);
            color: #ddd;
        }
        #svp-seekbar {
            flex-grow: 1;
            opacity: 0.5;
        }
        #svp-chapters {
            margin-top: 20px;
        }
        .svp-chapter {
            margin-right: 5px;
            padding: 5px;
            cursor: pointer;
            width: 6%;
            background-color: #202020;
            color: #fff;
        }
    </style>
    <?php
    return ob_get_clean();
}
add_shortcode('svp_player', 'svp_video_player_shortcode');
